<?php

namespace App\Jobs;

use App\Models\{
    ImportFile,
    ImportError,
    Device,
    Dealer,
    DeviceAllocation
};

use App\Imports\ChunkReadFilter;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use PhpOffice\PhpSpreadsheet\IOFactory;

class ProcessAllocationImport
{
    protected int $importFileId;

    public function __construct(int $importFileId)
    {
        $this->importFileId = $importFileId;
    }

    public function handle(): void
    {
        set_time_limit(0);
        ini_set('memory_limit', '1024M');

        $importFile = ImportFile::findOrFail($this->importFileId);

        if (!$importFile->file_path) {
            $importFile->update(['status' => 'failed']);
            return;
        }

        $filePath = Storage::disk('local')->path($importFile->file_path);

        if (!file_exists($filePath)) {
            $importFile->update(['status' => 'failed']);
            return;
        }

        $importFile->update([
            'status' => 'processing',
            'processed_rows' => 0,
        ]);

        $reader = IOFactory::createReaderForFile($filePath);
        $reader->setReadDataOnly(true);

        $info = $reader->listWorksheetInfo($filePath);
        $totalRows = max(((int) ($info[0]['totalRows'] ?? 0)) - 1, 0);

        $importFile->update(['total_rows' => $totalRows]);

        if ($totalRows === 0) {
            $importFile->update(['status' => 'completed']);
            return;
        }

        // 🚀 speed boost: preload dealers (NO query per row)
        $dealerMap = Dealer::where('tenant_id', $importFile->tenant_id)
            ->pluck('id', 'name')
            ->toArray();

        $chunkSize = 2000;
        $filter = new ChunkReadFilter();
        $reader->setReadFilter($filter);

        $processed = 0;

        for ($startRow = 2; $startRow <= ($totalRows + 1); $startRow += $chunkSize) {

            $filter->setRows($startRow, $chunkSize);
            $spreadsheet = $reader->load($filePath);
            $sheet = $spreadsheet->getActiveSheet();

            for ($row = $startRow; $row < ($startRow + $chunkSize) && $row <= ($totalRows + 1); $row++) {

                $dealerName = trim((string) $sheet->getCell("D{$row}")->getValue());
                $imei = trim(preg_replace('/\D/', '', (string) $sheet->getCell("H{$row}")->getFormattedValue()));

                $processed++; // ✅ always count for progress

                if ($dealerName === '' || $imei === '') {
                    $this->logError($importFile, $row, "Dealer/IMEI missing");
                    continue;
                }

                if (!isset($dealerMap[$dealerName])) {
                    $this->logError($importFile, $row, "Dealer not found: {$dealerName}");
                    continue;
                }

                try {
                    DB::transaction(function () use ($importFile, $dealerMap, $dealerName, $imei) {

                        $device = Device::where('tenant_id', $importFile->tenant_id)
                            ->where(function ($q) use ($imei) {
                                $q->where('imei_1', $imei)->orWhere('imei_2', $imei);
                            })
                            ->firstOrFail();

                        if ($device->status === 'activated') {
                            throw new \Exception('Already activated');
                        }

                        if (DeviceAllocation::where('device_id', $device->id)->exists()) {
                            throw new \Exception('Already allocated');
                        }

                        DeviceAllocation::create([
                            'tenant_id' => $importFile->tenant_id,
                            'device_id' => $device->id,
                            'dealer_id' => $dealerMap[$dealerName],
                            'import_file_id' => $importFile->id,
                            'allocated_at' => now(),
                        ]);

                        $device->update(['status' => 'allocated']);
                    });

                } catch (\Throwable $e) {
                    $this->logError($importFile, $row, $e->getMessage());
                }

                if ($processed % 200 === 0) {
                    ImportFile::where('id', $importFile->id)->update(['processed_rows' => $processed]);
                }
            }

            $spreadsheet->disconnectWorksheets();
            unset($spreadsheet);
        }

        $importFile->update([
            'processed_rows' => $totalRows,
            'status' => 'completed',
        ]);
    }

    private function logError(ImportFile $importFile, int $row, string $msg): void
    {
        ImportError::create([
            'import_file_id' => $importFile->id,
            'row_number' => $row,
            'error_message' => $msg,
            'created_by' => $importFile->uploaded_by,
        ]);
    }
}
