<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

use App\Models\ImportFile;
use App\Jobs\ProcessInventoryImport;
use App\Jobs\ProcessAllocationImport;
use App\Jobs\ProcessActivationImport;

class ImportController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | INVENTORY (MASTER) IMPORT  ✅ NEW CLEAN FLOW
    |--------------------------------------------------------------------------
    */
    public function inventory(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls,csv',
        ]);

        // Always store on LOCAL disk
        $storedPath = $request->file('file')->store('imports', 'local');

        DB::beginTransaction();

        try {
            $importFile = ImportFile::create([
                'tenant_id'         => Auth::user()->tenant_id,
                'uploaded_by'       => Auth::id(),
                'type'              => 'inventory',
                'original_filename' => $request->file('file')->getClientOriginalName(),
                'file_path'         => $storedPath,

                // 🔒 SAFE DEFAULTS (NO NULLS)
                'status'            => 'queued',
                'total_rows'        => 0,
                'processed_rows'    => 0,
            ]);

            DB::commit();

            // Dispatch AFTER commit
            ProcessInventoryImport::dispatch($importFile->id);

            return redirect()->route('imports.progress.view', $importFile);

        } catch (\Throwable $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /*
    |--------------------------------------------------------------------------
    | SALES TO DEALER (ALLOCATION) IMPORT  ✅ WORKING
    |--------------------------------------------------------------------------
    */
   public function allocation(Request $request)
{
    $request->validate([
        'file' => 'required|file|mimes:xlsx,xls,csv',
    ]);

    // 🚀 allow long execution for 30k+ rows
    set_time_limit(0);
    ini_set('memory_limit', '1024M');

    $storedPath = $request->file('file')->store('imports', 'local');

    DB::beginTransaction();

    try {
        $importFile = ImportFile::create([
            'tenant_id'         => Auth::user()->tenant_id,
            'uploaded_by'       => Auth::id(),
            'type'              => 'allocation',
            'original_filename' => $request->file('file')->getClientOriginalName(),
            'file_path'         => $storedPath,
            'status'            => 'processing',
            'total_rows'        => 0,
            'processed_rows'    => 0,
        ]);

        DB::commit();

        /**
         * 🔥 IMPORTANT FIX
         * Run import directly WITHOUT queue
         */
        (new \App\Jobs\ProcessAllocationImport($importFile->id))->handle();

        return redirect()->route('imports.progress.view', $importFile);

    } catch (\Throwable $e) {
        DB::rollBack();

        // mark failed safely
        if (isset($importFile)) {
            $importFile->update(['status' => 'failed']);
        }

        throw $e;
    }
}

   public function activation(Request $request)
{
    $request->validate([
        'file' => 'required|file|mimes:xlsx,xls,csv',
    ]);

    // 🔥 allow long execution for big Excel
    set_time_limit(0);
    ini_set('memory_limit', '1024M');

    $storedPath = $request->file('file')->store('imports', 'local');

    DB::beginTransaction();

    try {
        $importFile = ImportFile::create([
            'tenant_id'         => Auth::user()->tenant_id,
            'uploaded_by'       => Auth::id(),
            'type'              => 'activation',
            'original_filename' => $request->file('file')->getClientOriginalName(),
            'file_path'         => $storedPath,
            'status'            => 'processing', // ⭐ changed
            'total_rows'        => 0,
            'processed_rows'    => 0,
        ]);

        DB::commit();

        // 🔥 RUN DIRECTLY (NO QUEUE WORKER)
        app(\App\Jobs\ProcessActivationImport::class, [
            'importFileId' => $importFile->id
        ])->handle();

        return redirect()->route('imports.progress.view', $importFile);

    } catch (\Throwable $e) {
        DB::rollBack();
        throw $e;
    }
}

    /*
    |--------------------------------------------------------------------------
    | IMPORT PROGRESS (VIEW + API)
    |--------------------------------------------------------------------------
    */
    public function progressView(ImportFile $importFile)
    {
        return view('imports.progress', compact('importFile'));
    }

    public function progress(ImportFile $importFile)
    {
        return response()->json([
            'status'    => $importFile->status,
            'total'     => (int) $importFile->total_rows,
            'processed' => (int) $importFile->processed_rows,
        ]);
    }

    /*
    |--------------------------------------------------------------------------
    | UPLOAD SCREENS
    |--------------------------------------------------------------------------
    */
    public function showInventory()
    {
        return view('imports.inventory');
    }

    public function showAllocation()
    {
        return view('imports.allocation');
    }

    public function showActivation()
    {
        return view('imports.activation');
    }
}
